class InvalidGameFile(Exception):
    pass


class IllegalCategoryCount(Exception):
    pass


class IllegalPointsCount(Exception):
    pass


class GameTile:
    def __init__(self, category='', points=0, question='', answer=''):
        self.category = category  # The question category
        self.points = points  # The point value of this question
        self.question = question  # The question
        self.answer = answer  # The answer

    def __str__(self):  # This method creates a string representation of this object
        # Let's store all of our properties in a dict object
        result = {'category': self.category,
                  'points': self.points,
                  'question': self.question,
                  'answer': self.answer}

        # Now we can convert the dict to a string which will give us friendly formatting
        return str(result)

    def __repr__(self):
        # This method also creates a String representation of a Python object
        # The Python debugger calls this method rather than __str__
        # But we can just reuse our code by calling __str__
        return self.__str__()


class FinalJeopardyTile:
    def __init__(self, question='', answer=''):
        self.question = question
        self.answer = answer

    def __str__(self):
        result = {'question': self.question, 'answer': self.answer}
        return str(result)

    def __repr__(self):
        return self.__str__()


class GameBoard:
    def __init__(self, gameTiles):
        self.gameTiles = gameTiles


def readGameBoard(gameFile):
    # Read the entire file into memory
    rawLines = open(gameFile, 'r').readlines()

    # Check that we have 28 lines in the game file
    if len(rawLines) != 28:
        # Throw our custom exception to indicate the file is invalid
        raise InvalidGameFile

    # Now this is going to store our files
    fileLines = []
    for line in rawLines:  # Iterate through rawLines one item at a time
        # Now we need to remove any '\n' characters and store them in fileLines
        fileLines.append(line.strip())

    # There has to be 5 unique categories
    # We are going to split the first line apart by the :: character
    categories = fileLines[0].split('::')

    # Now check that we have 5 unique categories. We use the set() function that
    # will remove any duplicates and then we are going to check the length. It has to be
    # 5 per the requirements
    if len(set(categories)) != 5:
        raise IllegalCategoryCount  # Raise one of our custom exceptions

    # There has to be 5 unique points
    # So now we read the second line and split it one our :: character
    points = []  # Store the points to validate the file later
    pointValues = fileLines[1].split('::')

    # Now, we have Strings in pointValues, but we need them to be integers
    # So we iterate through pointValues and convert each item to an int
    # before storing it in points. We use the int() function to do the conversion
    for pv in pointValues:
        points.append(int(pv))

    # Now once again, we need to for 5 unique points so we use set again to remove
    # the duplicates and then len to check the size
    if len(set(points)) != 5:
        raise IllegalPointsCount  # Throw a custom exception

    gameTiles = {}  # Store all of the questions here

    # Now read everything else
    for line in fileLines[2:]:  # Slice off the first two lines
        #  Get a line and split it into its parts
        parts = line.split('::')

        #  Now we check that the category and points are valid
        if parts[0] in categories and int(parts[1]) in points:
            # We can create a GameTile object at this point
            gameTile = GameTile(category=parts[0],
                                points=int(parts[1]),
                                question=parts[2],
                                answer=parts[3])

            # If this our first insertion, we need to create a new list object to store our
            # GameTiles in our dictionary
            if parts[0] not in gameTiles:
                gameTiles[parts[0]] = []
            else:
                # Otherwise we can add our gameTile to gameTiles. Notice that the category is the key
                gameTiles[parts[0]].append(gameTile)

        # This handles the final jeopardy case
        elif parts[0] == 'FJ':
            # gameTiles uses FJ as the key for Final Jeopardy and then we store a FinalJeopardyTile
            gameTiles['FJ'] = FinalJeopardyTile(question=parts[1], answer=parts[2])
        else:
            # Throw our custom exception
            raise InvalidGameFile
    return gameTiles  # Return our dictionary that contains our question and final Jeopardy


if __name__ == '__main__':
    print(readGameBoard('questions.txt'))
